/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.dbpool;

import com.google.common.primitives.Ints;
import io.airlift.dbpool.ManagedDataSourceStats;
import io.airlift.dbpool.ManagedSemaphore;
import io.airlift.dbpool.SqlTimeoutException;
import io.airlift.units.Duration;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Logger;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.DataSource;
import javax.sql.PooledConnection;
import org.weakref.jmx.Flatten;
import org.weakref.jmx.Managed;

public abstract class ManagedDataSource
implements DataSource {
    private final ManagedSemaphore semaphore;
    private final AtomicInteger maxConnectionWaitMillis = new AtomicInteger(100);
    private final ManagedDataSourceStats stats = new ManagedDataSourceStats();

    protected ManagedDataSource(int maxConnections, Duration maxConnectionWait) {
        if (maxConnections < 1) {
            throw new IllegalArgumentException("maxConnections must be at least 1: maxConnections=" + maxConnections);
        }
        if (maxConnectionWait == null) {
            throw new NullPointerException("maxConnectionWait is null");
        }
        this.semaphore = new ManagedSemaphore(maxConnections);
        this.maxConnectionWaitMillis.set(Ints.checkedCast((long)maxConnectionWait.toMillis()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Connection getConnection() throws SQLException {
        long start = System.nanoTime();
        try {
            this.acquirePermit();
            boolean checkedOut = false;
            try {
                Connection connection = this.createConnection();
                checkedOut = true;
                Connection connection2 = connection;
                if (!checkedOut) {
                    this.semaphore.release();
                }
                return connection2;
            }
            catch (Throwable throwable) {
                if (!checkedOut) {
                    this.semaphore.release();
                }
                throw throwable;
            }
        }
        finally {
            this.stats.connectionCheckedOut(Duration.nanosSince((long)start));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected Connection createConnection() throws SQLException {
        boolean success = false;
        try {
            long start = System.nanoTime();
            PooledConnection pooledConnection = this.createConnectionInternal();
            Connection connection = this.prepareConnection(pooledConnection);
            this.stats.connectionCreated(Duration.nanosSince((long)start));
            success = true;
            Connection connection2 = connection;
            return connection2;
        }
        finally {
            if (!success) {
                this.stats.creationErrorOccurred();
            }
        }
    }

    protected abstract PooledConnection createConnectionInternal() throws SQLException;

    protected Connection prepareConnection(PooledConnection pooledConnection) throws SQLException {
        Connection connection = pooledConnection.getConnection();
        pooledConnection.addConnectionEventListener(new NoPoolConnectionEventListener());
        return connection;
    }

    protected void connectionReturned(PooledConnection pooledConnection, long checkoutTime) {
        try {
            pooledConnection.close();
        }
        catch (SQLException ignored) {
            // empty catch block
        }
    }

    protected void connectionDestroyed(PooledConnection pooledConnection, long checkoutTime) {
    }

    @Managed
    public int getMaxConnectionWaitMillis() {
        return this.maxConnectionWaitMillis.get();
    }

    @Managed
    public void setMaxConnectionWaitMillis(Duration maxConnectionWait) throws IllegalArgumentException {
        if (maxConnectionWait == null) {
            throw new NullPointerException("maxConnectionWait is null");
        }
        int millis = Ints.checkedCast((long)maxConnectionWait.toMillis());
        if (millis < 1) {
            throw new IllegalArgumentException("maxConnectionWait must be greater than 1 millisecond");
        }
        this.maxConnectionWaitMillis.set(millis);
    }

    @Managed
    public long getConnectionsActive() {
        return this.semaphore.getActivePermits();
    }

    @Managed
    public int getMaxConnections() {
        return this.semaphore.getPermits();
    }

    @Managed
    public void setMaxConnections(int maxConnections) {
        if (maxConnections < 1) {
            throw new IllegalArgumentException("maxConnections must be at least 1: maxConnections=" + maxConnections);
        }
        this.semaphore.setPermits(maxConnections);
    }

    @Managed
    @Flatten
    public ManagedDataSourceStats getStats() {
        return this.stats;
    }

    @Override
    public PrintWriter getLogWriter() throws SQLException {
        return null;
    }

    @Override
    public void setLogWriter(PrintWriter out) throws SQLException {
    }

    @Override
    public Logger getParentLogger() throws SQLFeatureNotSupportedException {
        throw new SQLFeatureNotSupportedException("java.util.logging not supported");
    }

    @Override
    public int getLoginTimeout() throws SQLException {
        return (int)Math.ceil((double)this.getMaxConnectionWaitMillis() / 1000.0);
    }

    @Override
    public void setLoginTimeout(int seconds) throws SQLException {
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        if (iface == null) {
            throw new SQLException("iface is null");
        }
        return iface.isInstance(this);
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        if (iface == null) {
            throw new SQLException("iface is null");
        }
        if (iface.isInstance(this)) {
            return iface.cast(this);
        }
        throw new SQLException(this.getClass().getName() + " does not implement " + iface.getName());
    }

    @Override
    public final Connection getConnection(String username, String password) throws SQLException {
        throw new UnsupportedOperationException();
    }

    private void acquirePermit() throws SQLException {
        int timeout = this.maxConnectionWaitMillis.get();
        try {
            if (!this.semaphore.tryAcquire(timeout, TimeUnit.MILLISECONDS)) {
                throw new SqlTimeoutException("Could not acquire a connection within " + timeout + " msec");
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new SqlTimeoutException("Interrupted while waiting for connection", e);
        }
    }

    protected class NoPoolConnectionEventListener
    implements ConnectionEventListener {
        private final long checkoutTime = System.nanoTime();
        private final AtomicBoolean returned = new AtomicBoolean();

        protected NoPoolConnectionEventListener() {
        }

        @Override
        public void connectionClosed(ConnectionEvent event) {
            if (!this.returned.compareAndSet(false, true)) {
                return;
            }
            PooledConnection pooledConnection = null;
            try {
                pooledConnection = (PooledConnection)event.getSource();
                pooledConnection.removeConnectionEventListener(this);
                ManagedDataSource.this.stats.connectionReturned(Duration.nanosSince((long)this.checkoutTime));
            }
            finally {
                ManagedDataSource.this.semaphore.release();
                if (pooledConnection != null) {
                    ManagedDataSource.this.connectionReturned(pooledConnection, this.checkoutTime);
                }
            }
        }

        @Override
        public void connectionErrorOccurred(ConnectionEvent event) {
            if (!this.returned.compareAndSet(false, true)) {
                return;
            }
            PooledConnection pooledConnection = null;
            try {
                pooledConnection = (PooledConnection)event.getSource();
                pooledConnection.removeConnectionEventListener(this);
                ManagedDataSource.this.stats.connectionErrorOccurred();
            }
            finally {
                ManagedDataSource.this.semaphore.release();
                if (pooledConnection != null) {
                    ManagedDataSource.this.connectionDestroyed(pooledConnection, this.checkoutTime);
                }
            }
        }
    }
}

