/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.vector;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.memory.OutOfMemoryRuntimeException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.hadoop.io.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;






/**
 * Decimal18 implements a vector of fixed width values.  Elements in the vector are accessed
 * by position, starting from the logical start of the vector.  Values should be pushed onto the
 * vector sequentially, but may be randomly accessed.
 *   The width of each element is 8 byte(s)
 *   The equivalent Java primitive is 'long'
 *
 * Source code generated using FreeMarker template FixedValueVectors.java
 */
@SuppressWarnings("unused")
public final class Decimal18Vector extends BaseDataValueVector implements FixedWidthVector{
  private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(Decimal18Vector.class);

  private final FieldReader reader = new Decimal18ReaderImpl(Decimal18Vector.this);
  private final Accessor accessor = new Accessor();
  private final Mutator mutator = new Mutator();

  private int allocationValueCount = INITIAL_VALUE_ALLOCATION;
  private int allocationMonitor = 0;

  public Decimal18Vector(MaterializedField field, BufferAllocator allocator) {
    super(field, allocator);
  }

  @Override
  public FieldReader getReader(){
    return reader;
  }

  public int getValueCapacity(){
    return (int) (data.capacity() *1.0 / 8);
  }

  public Accessor getAccessor(){
    return accessor;
  }

  public Mutator getMutator(){
    return mutator;
  }

  @Override
  public void setInitialCapacity(int numRecords) {
    allocationValueCount = numRecords;
  }

  public void allocateNew() {
    if(!allocateNewSafe()){
      throw new OutOfMemoryRuntimeException("Failure while allocating buffer.");
    }
  }

  public boolean allocateNewSafe() {
    clear();
    if (allocationMonitor > 10) {
      allocationValueCount = Math.max(8, (int) (allocationValueCount / 2));
      allocationMonitor = 0;
    } else if (allocationMonitor < -2) {
      allocationValueCount = (int) (allocationValueCount * 2);
      allocationMonitor = 0;
    }

    DrillBuf newBuf = allocator.buffer(allocationValueCount * 8);
    if(newBuf == null) {
      return false;
    }

    this.data = newBuf;
    this.data.readerIndex(0);
    return true;
  }

  /**
   * Allocate a new buffer that supports setting at least the provided number of values.  May actually be sized bigger depending on underlying buffer rounding size. Must be called prior to using the ValueVector.
   * @param valueCount
   * @throws org.apache.drill.exec.memory.OutOfMemoryRuntimeException if it can't allocate the new buffer
   */
  public void allocateNew(int valueCount) {
    clear();

    DrillBuf newBuf = allocator.buffer(valueCount * 8);
    if (newBuf == null) {
      throw new OutOfMemoryRuntimeException(
        String.format("Failure while allocating buffer of %d bytes",valueCount * 8));
    }

    this.data = newBuf;
    this.data.readerIndex(0);
    this.allocationValueCount = valueCount;
  }

/**
 * Allocate new buffer with double capacity, and copy data into the new buffer. Replace vector's buffer with new buffer, and release old one
 *
 * @throws org.apache.drill.exec.memory.OutOfMemoryRuntimeException if it can't allocate the new buffer
 */
  public void reAlloc() {
    logger.info("Realloc vector {}. [{}] -> [{}]", field, allocationValueCount * 8, allocationValueCount * 2 * 8);
    allocationValueCount *= 2;
    DrillBuf newBuf = allocator.buffer(allocationValueCount * 8);
    if (newBuf == null) {
      throw new OutOfMemoryRuntimeException(
      String.format("Failure while reallocating buffer to %d bytes",allocationValueCount * 8));
    }

    newBuf.setBytes(0, data, 0, data.capacity());
    newBuf.setZero(newBuf.capacity() / 2, newBuf.capacity() / 2);
    newBuf.writerIndex(data.writerIndex());
    data.release();
    data = newBuf;
  }

  public void zeroVector() {
    data.setZero(0, data.capacity());
  }

  @Override
  public int load(int valueCount, DrillBuf buf){
    clear();
    int len = valueCount * 8;
    data = buf.slice(0, len);
    data.retain();
    data.writerIndex(len);
    return len;
  }

  @Override
  public void load(SerializedField metadata, DrillBuf buffer) {
    assert this.field.matches(metadata) : String.format("The field %s doesn't match the provided metadata %s.", this.field, metadata);
    int loaded = load(metadata.getValueCount(), buffer);
    assert metadata.getBufferLength() == loaded : String.format("Expected to load %d bytes but actually loaded %d bytes", metadata.getBufferLength(), loaded);
  }

  public TransferPair getTransferPair(){
    return new TransferImpl(getField());
  }
  public TransferPair getTransferPair(FieldReference ref){
    return new TransferImpl(getField().withPath(ref));
  }

  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((Decimal18Vector) to);
  }

  public void transferTo(Decimal18Vector target){
    target.clear();
    target.data = data;
    target.data.retain();
    target.data.writerIndex(data.writerIndex());
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, Decimal18Vector target) {
    int currentWriterIndex = data.writerIndex();
    int startPoint = startIndex * 8;
    int sliceLength = length * 8;
    target.clear();
    target.data = this.data.slice(startPoint, sliceLength);
    target.data.writerIndex(sliceLength);
    target.data.retain();
  }

  private class TransferImpl implements TransferPair{
    Decimal18Vector to;

    public TransferImpl(MaterializedField field){
      this.to = new Decimal18Vector(field, allocator);
    }

    public TransferImpl(Decimal18Vector to) {
      this.to = to;
    }

    public Decimal18Vector getTo(){
      return to;
    }

    public void transfer(){
      transferTo(to);
    }

    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }

    @Override
    public void copyValueSafe(int fromIndex, int toIndex) {
      to.copyFromSafe(fromIndex, toIndex, Decimal18Vector.this);
    }
  }

  public void copyFrom(int fromIndex, int thisIndex, Decimal18Vector from){
 
    data.setLong(thisIndex * 8,
        from.data.getLong(fromIndex * 8)
    );
     
  }

  public void copyFromSafe(int fromIndex, int thisIndex, Decimal18Vector from){
    while(thisIndex >= getValueCapacity()) {
        reAlloc();
    }
    copyFrom(fromIndex, thisIndex, from);
  }

  public void decrementAllocationMonitor() {
    if (allocationMonitor > 0) {
      allocationMonitor = 0;
    }
    --allocationMonitor;
  }

  private void incrementAllocationMonitor() {
    ++allocationMonitor;
  }

  public final class Accessor extends BaseDataValueVector.BaseAccessor {

    public int getValueCount() {
      return data.writerIndex() / 8;
    }

    public boolean isNull(int index){
      return false;
    }

 

    public long get(int index) {
      return data.getLong(index * 8);
    }


    @Override
    public BigDecimal getObject(int index) {

        BigInteger value = BigInteger.valueOf(((Long)get(index)).longValue());
        return new BigDecimal(value, getField().getScale());
    }


    public void get(int index, Decimal18Holder holder){
      holder.scale = getField().getScale();
      holder.precision = getField().getPrecision();

      holder.value = data.getLong(index * 8);
    }

    public void get(int index, NullableDecimal18Holder holder){
      holder.isSet = 1;
      holder.value = data.getLong(index * 8);
    }


    
 }

 /**
  * Decimal18.Mutator implements a mutable vector of fixed width values.  Elements in the
  * vector are accessed by position from the logical start of the vector.  Values should be pushed
  * onto the vector sequentially, but may be randomly accessed.
  *   The width of each element is 8 byte(s)
  *   The equivalent Java primitive is 'long'
  *
  * NB: this class is automatically generated from ValueVectorTypes.tdd using FreeMarker.
  */
  public final class Mutator extends BaseDataValueVector.BaseMutator {

    private Mutator(){};
   /**
    * Set the element at the given index to the given value.  Note that widths smaller than
    * 32 bits are handled by the DrillBuf interface.
    *
    * @param index   position of the bit to set
    * @param value   value to set
    */
 
   public void set(int index, long value) {
     data.setLong(index * 8, value);
   }

   public void setSafe(int index, long value) {
     while(index >= getValueCapacity()) {
       reAlloc();
     }
     set(index, value);
   }

   protected void set(int index, Decimal18Holder holder){
     data.setLong(index * 8, holder.value);
   }

   public void setSafe(int index, Decimal18Holder holder){
     while(index >= getValueCapacity()) {
       reAlloc();
     }
     set(index, holder);
   }

   protected void set(int index, NullableDecimal18Holder holder){
     data.setLong(index * 8, holder.value);
   }

   public void setSafe(int index, NullableDecimal18Holder holder){
     while(index >= getValueCapacity()) {
       reAlloc();
     }
     set(index, holder);
   }

   @Override
   public void generateTestData(int size) {
     setValueCount(size);
     boolean even = true;
     for(int i =0; i < getAccessor().getValueCount(); i++, even = !even){
       if(even){
         set(i, Long.MIN_VALUE);
       }else{
         set(i, Long.MAX_VALUE);
       }
     }
   }


   public void generateTestDataAlt(int size) {
     setValueCount(size);
     boolean even = true;
     for(int i =0; i < getAccessor().getValueCount(); i++, even = !even){
       if(even){
         set(i, (long) 1);
       }else{
         set(i, (long) 0);
       }
     }
   }

   



   public void setValueCount(int valueCount) {
     int currentValueCapacity = getValueCapacity();
     int idx = (8 * valueCount);
     while(valueCount > getValueCapacity()) {
       reAlloc();
     }
     if (valueCount > 0 && currentValueCapacity > valueCount * 2) {
       incrementAllocationMonitor();
     } else if (allocationMonitor > 0) {
       allocationMonitor = 0;
     }
     VectorTrimmer.trim(data, idx);
     data.writerIndex(valueCount * 8);
   }





 }
}

 

