/*
 * Decompiled with CFR 0.152.
 */
package org.jodconverter.local.office;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.SystemUtils;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.jodconverter.core.office.OfficeException;
import org.jodconverter.core.office.RetryTimeoutException;
import org.jodconverter.local.office.ExitCodeRetryable;
import org.jodconverter.local.office.LocalOfficeUtils;
import org.jodconverter.local.office.OfficeDescriptor;
import org.jodconverter.local.office.OfficeUrl;
import org.jodconverter.local.office.StartProcessRetryable;
import org.jodconverter.local.office.VerboseProcess;
import org.jodconverter.local.process.LinesPumpStreamHandler;
import org.jodconverter.local.process.ProcessManager;
import org.jodconverter.local.process.ProcessQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OfficeProcess {
    private static final boolean DEFAULT_KILL_EXISTING_PROCESS = true;
    private static final long START_PROCESS_RETRY = 500L;
    private static final long START_PROCESS_TIMEOUT = 10000L;
    private static final Logger LOGGER = LoggerFactory.getLogger(OfficeProcess.class);
    private final OfficeUrl officeUrl;
    private final File officeHome;
    private final ProcessManager processManager;
    private final List<String> runAsArgs;
    private final File templateProfileDir;
    private final boolean killExistingProcess;
    private final File instanceProfileDir;
    private VerboseProcess process;
    private OfficeDescriptor descriptor;
    private long pid = -1L;

    public OfficeProcess(@NonNull OfficeUrl officeUrl, @NonNull File officeHome, @NonNull File workingDir, @NonNull ProcessManager processManager, @Nullable List<@NonNull String> runAsArgs, @Nullable File templateProfileDir, @Nullable Boolean killExistingProcess) {
        this.officeUrl = officeUrl;
        this.officeHome = officeHome;
        this.processManager = processManager;
        this.runAsArgs = runAsArgs == null ? Collections.emptyList() : Collections.unmodifiableList(runAsArgs);
        this.templateProfileDir = templateProfileDir;
        this.killExistingProcess = killExistingProcess == null ? true : killExistingProcess;
        this.instanceProfileDir = new File(workingDir, ".jodconverter_" + officeUrl.getConnectionAndParametersAsString().replace(',', '_').replace('=', '-'));
    }

    private void checkForExistingProcess(ProcessQuery processQuery) throws OfficeException {
        try {
            long existingPid = this.processManager.findPid(processQuery);
            if (existingPid > -1L && this.killExistingProcess) {
                LOGGER.warn("A process with --accept '{}' is already running; pid {}; trying to kill it...", (Object)processQuery.getArgument(), (Object)existingPid);
                this.processManager.kill(null, existingPid);
                this.waitForProcessToDie();
                existingPid = this.processManager.findPid(processQuery);
            }
            if (existingPid > -1L) {
                throw new OfficeException(String.format("A process with --accept '%s' is already running; pid %d", processQuery.getArgument(), existingPid));
            }
            LOGGER.debug("Checking existing process done successfully. No process running with --accept '{}'", (Object)processQuery.getArgument());
        }
        catch (IOException ioEx) {
            throw new OfficeException(String.format("Unable to check if there is already an existing process with --accept '%s'", processQuery.getArgument()), (Throwable)ioEx);
        }
    }

    public void deleteInstanceProfileDir() {
        LOGGER.debug("Deleting instance profile directory '{}'", (Object)this.instanceProfileDir);
        try {
            FileUtils.deleteDirectory((File)this.instanceProfileDir);
        }
        catch (IOException ioEx) {
            File oldProfileDir = new File(this.instanceProfileDir.getParentFile(), this.instanceProfileDir.getName() + ".old." + System.currentTimeMillis());
            if (this.instanceProfileDir.renameTo(oldProfileDir)) {
                LOGGER.warn("Could not delete profileDir; renamed it to '" + oldProfileDir + "'", (Throwable)ioEx);
            }
            LOGGER.error("Could not delete profileDir", (Throwable)ioEx);
        }
    }

    private void detectOfficeVersion() {
        File executable = LocalOfficeUtils.getOfficeExecutable(this.officeHome);
        ArrayList<String> command = new ArrayList<String>(this.runAsArgs);
        String execPath = executable.getAbsolutePath();
        this.descriptor = OfficeDescriptor.fromExecutablePath(execPath);
        if (SystemUtils.IS_OS_WINDOWS) {
            return;
        }
        String prefix = this.descriptor.useLongOptionNameGnuStyle() ? "--" : "-";
        command.add(execPath);
        command.add(prefix + "invisible");
        command.add(prefix + "help");
        command.add(prefix + "headless");
        command.add(prefix + "nocrashreport");
        command.add(prefix + "nodefault");
        command.add(prefix + "nofirststartwizard");
        command.add(prefix + "nolockcheck");
        command.add(prefix + "nologo");
        command.add(prefix + "norestore");
        command.add("-env:UserInstallation=" + LocalOfficeUtils.toUrl(this.instanceProfileDir));
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        try {
            Process process = processBuilder.start();
            LinesPumpStreamHandler handler = new LinesPumpStreamHandler(process.getInputStream(), process.getErrorStream());
            handler.start();
            try {
                process.waitFor();
                handler.stop();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            this.descriptor = OfficeDescriptor.fromHelpOutput(handler.getOutputPumper().getLines());
        }
        catch (IOException ioEx) {
            LOGGER.warn("An I/O error prevents us to determine office version", (Throwable)ioEx);
        }
    }

    public int forciblyTerminate(long retryInterval, long retryTimeout) throws OfficeException, RetryTimeoutException {
        if (this.process == null) {
            return 0;
        }
        LOGGER.info("Trying to forcibly terminate process: '{}'; pid: {}", (Object)this.officeUrl.getConnectionParametersAsString(), this.pid == -2L ? "PID_NOT_FOUND" : (this.pid == -1L ? "PID_UNKNOWN" : Long.valueOf(this.pid)));
        try {
            this.processManager.kill(this.process.getProcess(), this.pid);
        }
        catch (IOException ioEx) {
            throw new OfficeException("Unable to kill the process with pid: " + (this.pid == -2L ? "PID_NOT_FOUND" : (this.pid == -1L ? "PID_UNKNOWN" : Long.valueOf(this.pid))), (Throwable)ioEx);
        }
        return this.getExitCode(retryInterval, retryTimeout);
    }

    public @Nullable Integer getExitCode() {
        if (this.process == null) {
            return 0;
        }
        return this.process.getExitCode();
    }

    public int getExitCode(long retryInterval, long retryTimeout) throws OfficeException, RetryTimeoutException {
        if (this.process == null) {
            return 0;
        }
        try {
            ExitCodeRetryable retryable = new ExitCodeRetryable(this.process);
            retryable.execute(retryInterval, retryTimeout);
            return retryable.getExitCode();
        }
        catch (RetryTimeoutException timeoutEx) {
            throw timeoutEx;
        }
        catch (Exception ex) {
            throw new OfficeException("Could not get the process exit code", (Throwable)ex);
        }
    }

    public boolean isRunning() {
        return this.process != null && this.getExitCode() == null;
    }

    private void prepareInstanceProfileDir() throws OfficeException {
        if (this.instanceProfileDir.exists()) {
            LOGGER.warn("Profile dir '{}' already exists; deleting", (Object)this.instanceProfileDir);
            this.deleteInstanceProfileDir();
        }
        if (this.templateProfileDir != null) {
            try {
                FileUtils.copyDirectory((File)this.templateProfileDir, (File)this.instanceProfileDir);
            }
            catch (IOException ioEx) {
                throw new OfficeException("Failed to create the instance profile directory", (Throwable)ioEx);
            }
        }
    }

    private @NonNull ProcessBuilder prepareProcessBuilder(@NonNull String acceptString) {
        ArrayList<String> command = new ArrayList<String>(this.runAsArgs);
        File executable = LocalOfficeUtils.getOfficeExecutable(this.officeHome);
        String execPath = executable.getAbsolutePath();
        String prefix = this.descriptor.useLongOptionNameGnuStyle() ? "--" : "-";
        command.add(execPath);
        command.add(prefix + "accept=" + acceptString);
        command.add(prefix + "headless");
        command.add(prefix + "invisible");
        command.add(prefix + "nocrashreport");
        command.add(prefix + "nodefault");
        command.add(prefix + "nofirststartwizard");
        command.add(prefix + "nolockcheck");
        command.add(prefix + "nologo");
        command.add(prefix + "norestore");
        command.add("-env:UserInstallation=" + LocalOfficeUtils.toUrl(this.instanceProfileDir));
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("ProcessBuilder command: {}", (Object)String.join((CharSequence)" ", command));
        }
        return new ProcessBuilder(command);
    }

    public void start() throws OfficeException {
        this.start(false);
    }

    public void start(boolean restart) throws OfficeException {
        String acceptString = this.officeUrl.getConnectionAndParametersAsString() + ";" + this.officeUrl.getProtocolAndParametersAsString() + ";" + this.officeUrl.getRootOid();
        ProcessQuery processQuery = new ProcessQuery("soffice", acceptString);
        this.checkForExistingProcess(processQuery);
        if (!restart) {
            this.prepareInstanceProfileDir();
        }
        this.detectOfficeVersion();
        ProcessBuilder processBuilder = this.prepareProcessBuilder(acceptString);
        LOGGER.info("OFFICE HOME: {}", (Object)this.officeHome);
        LOGGER.info("Starting process with --accept '{}' and profileDir '{}'", (Object)acceptString, (Object)this.instanceProfileDir);
        try {
            StartProcessRetryable retryable = new StartProcessRetryable(this.processManager, processBuilder, processQuery);
            retryable.execute(500L, 10000L);
            this.process = retryable.getProcess();
            this.pid = retryable.getProcessId();
            LOGGER.info("Started process; pid: {}", this.pid == -2L ? "PID_NOT_FOUND" : (this.pid == -1L ? "PID_UNKNOWN" : Long.valueOf(this.pid)));
        }
        catch (OfficeException officeEx) {
            throw officeEx;
        }
        catch (Exception ex) {
            throw new OfficeException(String.format("An error prevents us to start a process with --accept '%s'", processQuery.getArgument()), (Throwable)ex);
        }
        if (this.pid == -2L) {
            throw new OfficeException(String.format("A process with --accept '%s' started but its pid could not be found", acceptString));
        }
    }

    private void waitForProcessToDie() {
        try {
            Thread.sleep(1000L);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
    }
}

