/*
 * Copyright (c) 2006-2013 Rogério Liesenfeld
 * This file is subject to the terms of the MIT license (see LICENSE.txt).
 */
package mockit;

import java.lang.reflect.*;

import mockit.internal.annotations.*;
import mockit.internal.startup.*;

/**
 * Provides static methods for the mocking and stubbing of arbitrary classes, according to specified <em>mock
 * classes</em> defined in test code.
 * Such methods are intended to be called from test code only.
 * <p/>
 * Once mocked, a "real" method defined in a production class will behave (during test execution) as if its
 * implementation was replaced by a call to the corresponding <em>mock method</em> in the mock class.
 * Whatever value this mock method returns will be the value returned by the call to the mocked method.
 * The mock method can also throw an exception or error, which will then be propagated to the caller of the mocked
 * "real" method.
 * Therefore, while mocked the original code in the real method is never executed (actually, there's still a way to
 * execute the real implementation, although not normally used for testing purposes).
 * The same basic rules apply to constructors, which can be mocked by mock methods with the special name "$init".
 * <p/>
 * The methods in this class can be divided in the following groups:
 * <ul>
 * <li>
 * <strong>Stubbing API</strong>: {@link #stubOut(Class...)}, {@link #stubOutClass(Class, String...)}, and
 * {@link #stubOutClass(Class, boolean, String...)}.
 * </li>
 * <li>
 * <strong>Mockups API</strong> for state-oriented mocking: {@link MockUp}, {@link #setUpMocks(Object...)}, and
 * {@link #setUpMock(Class, Class)} and its several overloads.
 * </li>
 * </ul>
 * In the Tutorial:
 * <a href="http://jmockit.googlecode.com/svn/trunk/www/tutorial/StateBasedTesting.html">Writing state-based tests</a>
 */
@SuppressWarnings("deprecation")
public final class Mockit
{
   static { Startup.verifyInitialization(); }
   private Mockit() {}

   /**
    * Stubs out all methods, constructors, and static initializers in the given classes, so that they do nothing
    * whenever executed.
    * <p/>
    * Note that any stubbed out constructor will still call a constructor in the super-class, which in turn will be
    * executed normally unless also stubbed out.
    * The super-constructor to be called is chosen arbitrarily.
    * The classes are stubbed out in the order they are given, so make sure any super-class comes first.
    * <p/>
    * Methods with non-<code>void</code> return type will return the default value for this type, that is, zero for a
    * number or {@code char}, {@literal false} for a boolean, empty for an array, or {@literal null} for a reference
    * type.
    * <p/>
    * If a different behavior is desired for any method or constructor, then {@link #setUpMocks(Object...)} and the
    * other similar methods can be used right after the call to this method.
    * They will override any stub previously created with the corresponding mock implementation, if any.
    *
    * @param realClasses one or more regular classes to be stubbed out
    *
    * @deprecated Use the {@link UsingMocksAndStubs} annotation instead.
    */
   @Deprecated
   public static void stubOut(Class<?>... realClasses)
   {
      for (Class<?> realClass : realClasses) {
         new ClassStubbing(realClass).stubOut();
      }
   }

   /**
    * Same as {@link #stubOut(Class...)} for the given class, except that only the specified class members (if any) are
    * stubbed out, leaving the rest unaffected.
    * Such class members include the methods and constructors defined by the class, plus any static or instance
    * initialization blocks it might have.
    * Note that if <em>no</em> filters are specified the whole class will be stubbed out.
    * <p/>
    * For methods, the filters are {@linkplain java.util.regex.Pattern regular expressions} for method names, optionally
    * followed by parameter type names between parentheses. For constructors, only the parameters are specified.
    * For more details about the syntax for mock filters, see the {@link MockClass#stubs} annotation attribute.
    * <p/>
    * The special filter "&lt;clinit>" will match all static initializers in the given class.
    * <p/>
    * To stub out instance field initializers it is necessary to actually specify all constructors in the class, because
    * such initialization assignments are copied to each and every constructor by the Java compiler.
    *
    * @param realClass a regular class to be stubbed out
    * @param filters one or more filters that specify which class members (methods, constructors, and/or static
    * initialization blocks) to be stubbed out
    *
    * @deprecated Use {@link MockUp} instead, with empty {@link Mock} methods for the methods and constructors to be
    * stubbed out.
    */
   @Deprecated
   public static void stubOutClass(Class<?> realClass, String... filters)
   {
      new ClassStubbing(realClass, true, filters).stubOut();
   }

   /**
    * The same as {@link #stubOutClass(Class, String...)}, but specifying whether filters are to be inverted or not.
    *
    * @param inverse indicates whether the mock filters are to be inverted or not; if inverted, only the methods and
    * constructors matching them are <strong>not</strong> mocked
    *
    * @deprecated Use {@link MockUp} instead, with empty {@link Mock} methods for the methods and constructors to be
    * stubbed out.
    */
   @Deprecated
   public static void stubOutClass(Class<?> realClass, boolean inverse, String... filters)
   {
      new ClassStubbing(realClass, !inverse, filters).stubOut();
   }

   /**
    * Sets up the mocks defined in one or more {@linkplain MockClass mock classes}.
    * <p/>
    * After this call, all such mocks are "in effect" until the end of the test method inside which it appears, if this
    * is the case.
    * If the method is a "before"/"setUp" method which executes before all test methods, then the mocks will remain in
    * effect until the end of the test (including any "after"/"tearDown" methods).
    * <p/>
    * Any invocation count constraints specified on mock methods (such as {@code @Mock(invocations = 1)}, for example)
    * will be automatically verified after the code under test is executed.
    * <p/>
    * For each call made during test execution to a <em>mocked</em> method, the corresponding <em>mock</em> method is
    * called instead.
    * A mock method must have the same signature (ie, name and parameters) as the corresponding mocked/real method.
    * The return type of the mock method must be the same exact type <em>or</em> a compatible one.
    * The {@code throws} clause may differ in any way.
    * Note also that the mock method can be static or not, independently of the real method being static or not.
    * <p/>
    * A constructor in the real class can be mocked by a corresponding mock method of name {@code $init}, declared
    * with the same parameters and with {@code void} return type.
    * It will be called for each new instance of the real class that is created through a call to that constructor, with
    * whatever arguments are passed to it.
    * <p/>
    * <strong>Class initializers</strong> of the real class (one or more {@code static} initialization blocks plus all
    * assignments to {@code static} fields) can be mocked by providing a mock method named {@code $clinit} in the mock
    * class. This method should return {@code void} and have no declared parameters.
    * It will be called at most once, at the time the real class is initialized by the JVM (and since all static
    * initializers for that class are mocked, the initialization will have no effect).
    * <p/>
    * Mock methods can gain access to the instance of the real class on which the corresponding real method or
    * constructor was called. This requires the mock class to define an instance field of name <strong>"it"</strong>,
    * the same type as the real class, and accessible from that class (in general, this means the field will have to be
    * {@code public}). Such a field will always be set to the appropriate real class instance, whenever a mock method is
    * called. Note that through this field the mock class will be able to call any accessible instance method on the
    * real class, including the real method corresponding to the current mock method. In this case, however, such calls
    * are not allowed by default because they lead to infinite recursion, with the mock calling itself indirectly
    * through the redefined real method. If the real method needs to be called from the mock method, then the latter
    * must be declared as {@linkplain mockit.Mock#reentrant reentrant}.
    *
    * @param mockClassesOrInstances one or more classes ({@code Class} objects) or instances of classes which define
    * arbitrary methods and/or constructors, where the ones annotated as {@linkplain Mock mocks} will be used to
    * redefine corresponding real methods/constructors in a designated {@linkplain MockClass#realClass() real class}
    * (usually, a class on which the code under test depends on)
    *
    * @throws IllegalArgumentException if a given mock class fails to specify the corresponding real class using the
    * {@code @MockClass(realClass = ...)} annotation; or if a mock class defines a mock method for which no
    * corresponding real method or constructor exists in the real class;
    * or if the real method matching a mock method is {@code abstract}
    *
    * @deprecated Use {@link MockUp} instead.
    */
   @Deprecated
   public static void setUpMocks(Object... mockClassesOrInstances)
   {
      for (Object mockClassOrInstance : mockClassesOrInstances) {
         Class<?> mockClass;
         Object mock;

         if (mockClassOrInstance instanceof Class<?>) {
            mockClass = (Class<?>) mockClassOrInstance;
            mock = null;
         }
         else {
            mockClass = mockClassOrInstance.getClass();
            mock = mockClassOrInstance;
         }

         new MockClassSetup(mock, mockClass).redefineMethods();
      }
   }

   /**
    * Similar to {@link #setUpMocks(Object...)}, but accepting a single mock and its corresponding real class.
    * <p/>
    * Useful when the real class is not known in advance, such as when it is determined at runtime through configuration
    * of by creating a {@link Proxy} for an interface.
    *
    * @param realClass the class to be mocked that is used by code under test
    * @param mock an instance of the class containing the mock methods for the real class
    *
    * @see
    * <a href="http://code.google.com/p/jmockit/source/browse/trunk/samples/orderMngmntWebapp/test/orderMngr/domain/order/OrderRepository_MockupsAPI_Test.java#113">
    * Example</a>
    *
    * @deprecated Use {@link MockUp#MockUp()} or {@link MockUp#MockUp(Class)} instead.
    */
   @Deprecated
   public static void setUpMock(Class<?> realClass, Object mock)
   {
      Class<?> mockClass = mock.getClass();
      new MockClassSetup(realClass, mock, mockClass).redefineMethods();
   }

   /**
    * Similar to {@link #setUpMocks(Object...)}, but accepting a single mock class and its corresponding real class.
    * <p/>
    * Can also be useful when the real class is not known in advance, such as when it is determined at runtime through
    * configuration or by creating a {@link Proxy} for an interface.
    *
    * @param realClass the class to be mocked that is used by code under test
    * @param mockClass the class containing the mock methods for the real class
    *
    * @see
    * <a href="http://code.google.com/p/jmockit/source/browse/trunk/main/test/mockit/MockAnnotationsTest.java#169">
    * Example</a>
    *
    * @deprecated Use {@link MockUp#MockUp()} or {@link MockUp#MockUp(Class)} instead.
    */
   @Deprecated
   public static void setUpMock(Class<?> realClass, Class<?> mockClass)
   {
      new MockClassSetup(realClass, null, mockClass).redefineMethods();
   }

   /**
    * Sets up the mocks defined in the given mock class.
    * <p/>
    * If the type {@linkplain MockClass#realClass referred to} by the mock class is actually an interface, then a new
    * empty implementation class is created.
    *
    * @param mockClassOrInstance the mock class itself (given by its {@code Class} literal), or an instance of the mock
    * class
    *
    * @return a new instance of the implementation class created for the mocked interface, or {@code null} otherwise
    *
    * @throws IllegalArgumentException if a given mock class fails to specify the corresponding real class using the
    * {@code @MockClass(realClass = ...)} annotation; or if a mock class defines a mock method for which no
    * corresponding real method or constructor exists in the real class;
    * or if the real method matching a mock method is {@code abstract}
    *
    * @see #setUpMock(Class, Object)
    * @see #setUpMocks(Object...)
    * @see
    * <a href="http://code.google.com/p/jmockit/source/browse/trunk/main/test/mockit/MockAnnotationsTest.java#696">
    * Example</a>
    *
    * @deprecated Use {@link MockUp#MockUp()} and {@link MockUp#getMockInstance()} (when mocking an interface) instead.
    */
   @Deprecated
   public static <T> T setUpMock(Object mockClassOrInstance)
   {
      Class<?> mockClass;
      Object mock;

      if (mockClassOrInstance instanceof Class<?>) {
         mockClass = (Class<?>) mockClassOrInstance;
         mock = null;
      }
      else {
         mockClass = mockClassOrInstance.getClass();
         mock = mockClassOrInstance;
      }

      Class<T> realClass = MockClassSetup.getRealClass(mockClass);

      if (realClass.isInterface()) {
         return new MockedImplementationClass<T>(mockClass, mock).generate(realClass, null);
      }

      new MockClassSetup(realClass, mock, mockClass).redefineMethods();
      return null;
   }
}
