
/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.expr.fn.impl;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.memory.OutOfMemoryRuntimeException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.hadoop.io.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;






import org.apache.drill.exec.expr.DrillSimpleFunc;
import org.apache.drill.exec.expr.annotations.FunctionTemplate;
import org.apache.drill.exec.expr.annotations.FunctionTemplate.NullHandling;
import org.apache.drill.exec.expr.annotations.Output;
import org.apache.drill.exec.expr.annotations.Param;
import org.apache.drill.exec.expr.annotations.Workspace;
import org.apache.drill.exec.expr.fn.FunctionGenerationHelper;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.exec.record.RecordBatch;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.DrillBuf;

import java.nio.ByteBuffer;

@SuppressWarnings("unused")
public class Decimal38SparseFunctions {

    @FunctionTemplate(name = "subtract", scope = FunctionTemplate.FunctionScope.DECIMAL_ADD_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseSubtractFunction implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Inject DrillBuf buffer;
        @Workspace int outputScale;
        @Workspace int outputPrecision;
        @Output Decimal38SparseHolder result;

        public void setup() {
            int size = (6 * (org.apache.drill.exec.util.DecimalUtility.integerSize));
            buffer = buffer.reallocIfNeeded(size);
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionAddFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionAddFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }
            result.precision = outputPrecision;
            result.scale = outputScale;
            result.buffer = buffer;
            result.start = 0;

            java.math.BigDecimal leftInput = org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromSparse(left.buffer, left.start, left.nDecimalDigits, left.scale);
            java.math.BigDecimal rightInput = org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromSparse(right.buffer, right.start, right.nDecimalDigits, right.scale);
            java.math.BigDecimal addResult = leftInput.subtract(rightInput);

            // Set the scale
            addResult.setScale(result.scale, java.math.BigDecimal.ROUND_HALF_UP);
            org.apache.drill.exec.util.DecimalUtility.getSparseFromBigDecimal(addResult, result.buffer, result.start, result.scale, result.precision, result.nDecimalDigits);
        }
    }

    @FunctionTemplate(name = "add", scope = FunctionTemplate.FunctionScope.DECIMAL_ADD_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseAddFunction implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Workspace int outputScale;
        @Workspace int outputPrecision;
        @Inject DrillBuf buffer;
        @Output Decimal38SparseHolder result;

        public void setup() {
            int size = (6 * (org.apache.drill.exec.util.DecimalUtility.integerSize));
            buffer = buffer.reallocIfNeeded(size);
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionAddFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionAddFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }
            result.precision = outputPrecision;
            result.scale = outputScale;
            result.buffer = buffer;
            result.start = 0;

            java.math.BigDecimal leftInput = org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromSparse(left.buffer, left.start, left.nDecimalDigits, left.scale);
            java.math.BigDecimal rightInput = org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromSparse(right.buffer, right.start, right.nDecimalDigits, right.scale);
            java.math.BigDecimal addResult = leftInput.add(rightInput);

            // Set the scale
            addResult.setScale(result.scale, java.math.BigDecimal.ROUND_HALF_UP);
            org.apache.drill.exec.util.DecimalUtility.getSparseFromBigDecimal(addResult, result.buffer, result.start, result.scale, result.precision, result.nDecimalDigits);
        }
    }

    @FunctionTemplate(name = "multiply", scope = FunctionTemplate.FunctionScope.DECIMAL_MUL_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseMultiplyFunction implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Inject DrillBuf buffer;
        @Workspace int[] tempResult;
        @Workspace int outputScale;
        @Workspace int outputPrecision;
        @Output Decimal38SparseHolder result;

        public void setup() {
            int size = (6 * (org.apache.drill.exec.util.DecimalUtility.integerSize));
            buffer = buffer.reallocIfNeeded(size);
            tempResult = new int[6 * 6];
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionMulFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionMulFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }
            // Set the scale and precision
            result.scale = outputScale;
            result.precision = outputPrecision;
            result.buffer = buffer;
            result.start = 0;

            // Re initialize the temporary array
            for (int i = 0; i < 6 * 6; i++) {
                tempResult[i] = 0;
            }

            // Remove the leading zeroes from the integer part of the input
            int leftIndex = 0;
            int leftStopIndex = left.nDecimalDigits - org.apache.drill.exec.util.DecimalUtility.roundUp(left.scale);

            while (leftIndex < leftStopIndex) {
                if (left.getInteger(leftIndex, left.start, left.buffer) > 0)
                    break;
                leftIndex++;
            }

            int leftIntegerSize = leftStopIndex - leftIndex;

            /* Remove the leading zeroes from the integer part of the input */
            int rightIndex = 0;
            int rightStopIndex = right.nDecimalDigits - org.apache.drill.exec.util.DecimalUtility.roundUp(right.scale);

            while(rightIndex < rightStopIndex) {
                if (right.getInteger(rightIndex, right.start, right.buffer) > 0)
                    break;
                rightIndex++;
            }

            int rightIntegerSize = rightStopIndex - rightIndex;

            int resultIntegerSize = leftIntegerSize + rightIntegerSize;
            int resultScaleSize = org.apache.drill.exec.util.DecimalUtility.roundUp(left.scale + right.scale);

            int leftSize  = left.nDecimalDigits - 1;
            int rightSize = right.nDecimalDigits - 1;

            int resultIndex = tempResult.length - 1;
            int currentIndex = 0;

            for (int i = leftSize; i >= leftIndex; i--) {

                currentIndex = resultIndex;
                int carry = 0;

                for (int j = rightSize; j >= rightIndex; j--) {

                    long mulResult = (long) right.getInteger(j, right.start, right.buffer) * (long) left.getInteger(i, left.start, left.buffer);

                    long tempSum = tempResult[currentIndex] + mulResult + carry;

                    if (tempSum >= org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE) {
                        tempResult[currentIndex] = (int) (tempSum % org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE);
                        carry = (int) (tempSum / org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE);
                    } else {
                        tempResult[currentIndex] = (int) tempSum;
                        carry = 0;
                    }

                    currentIndex--;
                }
                /* Propagate the carry */
                if (carry > 0)
                    tempResult[currentIndex] += carry;

                resultIndex--;
            }

            /* We have computed the result of the multiplication; check if we need to
             * round a portion of the fractional part
             */
            resultScaleSize = org.apache.drill.exec.util.DecimalUtility.roundUp(result.scale);

            if (result.scale < (left.scale + right.scale)) {
              /* The scale of the output data type is less than the scale
               * we obtained as a result of multiplication, we need to round
               * a chunk of the fractional part
               */
              int lastScaleIndex = currentIndex + resultIntegerSize + resultScaleSize - 1;

              // Compute the power of 10 necessary to find if we need to round up
              int roundFactor = (int) (org.apache.drill.exec.util.DecimalUtility.getPowerOfTen(
                                        org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS - ((result.scale + 1) % org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS)));

              // Index of rounding digit
              int roundIndex = currentIndex + resultIntegerSize + org.apache.drill.exec.util.DecimalUtility.roundUp(result.scale + 1) - 1;

              // Check the first chopped digit to see if we need to round up
              int carry = ((tempResult[roundIndex] / roundFactor) % 10) > 4 ? 1 : 0;

              if (result.scale > 0) {

                // Compute the power of 10 necessary to chop of the fractional part
                int scaleFactor = (int) (org.apache.drill.exec.util.DecimalUtility.getPowerOfTen(
                                         org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS - (result.scale % org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS)));
                // Chop the unwanted fractional part
                tempResult[lastScaleIndex] /=  scaleFactor;
                tempResult[lastScaleIndex] *= scaleFactor;

                // Adjust the carry so that it gets added to the correct digit
                carry *= scaleFactor;
              }

              // Propagate the carry
              while (carry > 0 && lastScaleIndex >= 0) {
                int tempSum = tempResult[lastScaleIndex] + carry;
                if (tempSum >= org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE) {
                  tempResult[lastScaleIndex] = (tempSum % org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE);
                  carry = (int) (tempSum / org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE);
                } else {
                  tempResult[lastScaleIndex] = tempSum;
                  carry = 0;
                }
                lastScaleIndex--;
              }

              // Check if carry has increased integer digit
              if ((lastScaleIndex + 1) < currentIndex) {
                resultIntegerSize++;
                currentIndex = lastScaleIndex + 1;
              }
            }

            if (resultIntegerSize > result.nDecimalDigits) {
              throw new org.apache.drill.common.exceptions.DrillRuntimeException("Cannot fit multiplication result in the given decimal type");
            }

            int outputIndex = result.nDecimalDigits - 1;

            for (int i = (currentIndex + resultIntegerSize + resultScaleSize - 1); i >= currentIndex; i--) {
                result.setInteger(outputIndex--, tempResult[i], result.start, result.buffer);
            }

            // Set the remaining digits to be zero
            while(outputIndex >= 0) {
              result.setInteger(outputIndex--, 0, result.start, result.buffer);
            }
            result.setSign(left.getSign(left.start, left.buffer) != right.getSign(right.start, right.buffer), result.start, result.buffer);
        }
    }

    @FunctionTemplate(name = "exact_divide", scope = FunctionTemplate.FunctionScope.DECIMAL_DIV_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseDivideFunction implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Output Decimal38SparseHolder result;
        @Inject DrillBuf buffer;
        @Workspace int outputScale;
        @Workspace int outputPrecision;

        public void setup() {
            int size = (6 * (org.apache.drill.exec.util.DecimalUtility.integerSize));
            buffer = buffer.reallocIfNeeded(size);
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionDivideFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionDivideFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }
            result.scale = outputScale;
            result.precision = outputPrecision;
            result.buffer = buffer;
            result.start = 0;

            java.math.BigDecimal numerator = org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromDrillBuf(left.buffer, left.start, left.nDecimalDigits, left.scale, true);
            java.math.BigDecimal denominator = org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromDrillBuf(right.buffer, right.start, right.nDecimalDigits, right.scale, true);

            java.math.BigDecimal output = numerator.divide(denominator, (int) result.scale, java.math.BigDecimal.ROUND_HALF_UP);

            org.apache.drill.exec.util.DecimalUtility.getSparseFromBigDecimal(output, result.buffer, result.start, result.scale, result.precision, result.nDecimalDigits);
        }
    }

    @FunctionTemplate(name = "mod", scope = FunctionTemplate.FunctionScope.DECIMAL_MOD_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseModFunction implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Output Decimal38SparseHolder result;
        @Inject DrillBuf buffer;
        @Workspace int outputScale;
        @Workspace int outputPrecision;

        public void setup() {
            int size = (6 * (org.apache.drill.exec.util.DecimalUtility.integerSize));
            buffer = buffer.reallocIfNeeded(size);
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionModFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionModFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }
            result.scale = outputScale;
            result.precision = outputPrecision;
            result.buffer = buffer;
            result.start = 0;

            java.math.BigDecimal numerator = org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromDrillBuf(left.buffer, left.start, left.nDecimalDigits, left.scale, true);
            java.math.BigDecimal denominator = org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromDrillBuf(right.buffer, right.start, right.nDecimalDigits, right.scale, true);

            java.math.BigDecimal output = numerator.remainder(denominator);
            output.setScale(result.scale, java.math.BigDecimal.ROUND_HALF_UP);

            org.apache.drill.exec.util.DecimalUtility.getSparseFromBigDecimal(output, result.buffer, result.start, result.scale, result.precision, result.nDecimalDigits);
        }
    }

    @FunctionTemplate(name = "abs", scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseAbsFunction implements DrillSimpleFunc {

        @Param  Decimal38SparseHolder in;
        @Output Decimal38SparseHolder out;
        public void setup() {

        }

        public void eval() {
          out.scale = in.scale;
          out.precision = in.precision;
          out.buffer = in.buffer;
          out.start = in.start;

          // Set the output buffer with the positive sign
          out.buffer.setInt(out.start, (out.buffer.getInt(out.start) & 0x7fffffff));
        }
    }

    @FunctionTemplate(name = "sign", scope = FunctionTemplate.FunctionScope.SIMPLE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseSignFunction implements DrillSimpleFunc {

        @Param Decimal38SparseHolder in;
        @Output IntHolder out;

        public void setup() {}

        public void eval() {

          boolean zeroValue = true;

          if (in.getSign(in.start, in.buffer) == true) {
            out.value = -1;
          } else {
            for (int i = 0; i < 6; i++) {
              if (in.getInteger(i, in.start, in.buffer) != 0) {
                zeroValue = false;
                break;
              }
            }
            out.value = (zeroValue == true) ? 0 : 1;
          }
        }
    }

    @FunctionTemplate(names = {"ceil", "ceiling"}, scope = FunctionTemplate.FunctionScope.DECIMAL_ZERO_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseCeilFunction implements DrillSimpleFunc {

        @Param  Decimal38SparseHolder in;
        @Output Decimal38SparseHolder out;
        public void setup() {

        }

        public void eval() {
          out.scale = 0;
          out.precision = in.precision;
          out.buffer = in.buffer;
          out.start = in.start;
          boolean sign = in.getSign(in.start, in.buffer);

          // Indicates whether we need to add 1 to the integer part, while performing ceil
          int carry = 0;

          int scaleStartIndex = 6 - org.apache.drill.exec.util.DecimalUtility.roundUp(in.scale);
          int srcIntIndex = scaleStartIndex - 1;

          if (sign == false) {
            // For negative values ceil we don't need to increment the integer part
            while (scaleStartIndex < 6) {
              if (out.getInteger(scaleStartIndex, out.start, out.buffer) != 0) {
                carry = 1;
                break;
              }
              scaleStartIndex++;
            }
          }

          // Truncate the fractional part, move the integer part
          int destIndex = 6 - 1;
          while (srcIntIndex >= 0) {
            out.setInteger(destIndex--, out.getInteger(srcIntIndex--, out.start, out.buffer), out.start, out.buffer);
          }

          // Set the remaining portion of the decimal to be zeroes
          while (destIndex >= 0) {
            out.setInteger(destIndex--, 0, out.start, out.buffer);
          }

          // Add the carry
          if (carry != 0) {
            destIndex = 6 - 1;

            while (destIndex >= 0) {
              int intValue = out.getInteger(destIndex, out.start, out.buffer);
              intValue += carry;

              if (intValue >= org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE) {
                out.setInteger(destIndex--, intValue % org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE, out.start, out.buffer);
                carry = intValue / org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE;
              } else {
                out.setInteger(destIndex--, intValue, out.start, out.buffer);
                break;
              }
            }
          }
          // Set the sign
          out.setSign(sign, out.start, out.buffer);
        }
    }

    @FunctionTemplate(name = "floor", scope = FunctionTemplate.FunctionScope.DECIMAL_ZERO_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseFloorFunction implements DrillSimpleFunc {

        @Param  Decimal38SparseHolder in;
        @Output Decimal38SparseHolder out;
        public void setup() {

        }

        public void eval() {
          out.scale = 0;
          out.precision = in.precision;
          out.buffer = in.buffer;
          out.start = in.start;
          boolean sign = in.getSign(in.start, in.buffer);

          // Indicates whether we need to decrement 1 from the integer part, while performing floor, done for -ve values
          int carry = 0;

          int scaleStartIndex = 6 - org.apache.drill.exec.util.DecimalUtility.roundUp(in.scale);
          int srcIntIndex = scaleStartIndex - 1;

          if (sign == true) {
            // For negative values ceil we don't need to increment the integer part
            while (scaleStartIndex < 6) {
              if (out.getInteger(scaleStartIndex, out.start, out.buffer) != 0) {
                carry = 1;
                break;
              }
              scaleStartIndex++;
            }
          }

          // Truncate the fractional part, move the integer part
          int destIndex = 6 - 1;
          while (srcIntIndex >= 0) {
            out.setInteger(destIndex--, out.getInteger(srcIntIndex--, out.start, out.buffer), out.start, out.buffer);
          }

          // Set the remaining portion of the decimal to be zeroes
          while (destIndex >= 0) {
            out.setInteger(destIndex--, 0, out.start, out.buffer);
          }
          // Add the carry
          if (carry != 0) {
            destIndex = 6 - 1;

            while (destIndex >= 0) {
              int intValue = out.getInteger(destIndex, out.start, out.buffer);
              intValue += carry;

              if (intValue >= org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE) {
                out.setInteger(destIndex--, intValue % org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE, out.start, out.buffer);
                carry = intValue / org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE;
              } else {
                out.setInteger(destIndex--, intValue, out.start, out.buffer);
                break;
              }
            }
          }
          // Set the sign
          out.setSign(sign, out.start, out.buffer);
        }
    }

    @FunctionTemplate(names = {"trunc", "truncate"}, scope = FunctionTemplate.FunctionScope.DECIMAL_ZERO_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseTruncateFunction implements DrillSimpleFunc {

        @Param  Decimal38SparseHolder in;
        @Output Decimal38SparseHolder out;
        public void setup() {

        }

        public void eval() {
          out.scale = 0;
          out.precision = in.precision;
          out.buffer = in.buffer;
          out.start = in.start;
          boolean sign = in.getSign(in.start, in.buffer);

          // Integer part's src index
          int srcIntIndex = 6 - org.apache.drill.exec.util.DecimalUtility.roundUp(in.scale) - 1;

          // Truncate the fractional part, move the integer part
          int destIndex = 6 - 1;
          while (srcIntIndex >= 0) {
            out.setInteger(destIndex--, out.getInteger(srcIntIndex--, out.start, out.buffer), out.start, out.buffer);
          }

          // Set the remaining portion of the decimal to be zeroes
          while (destIndex >= 0) {
            out.setInteger(destIndex--, 0, out.start, out.buffer);
          }
          // Set the sign
          out.setSign(sign, out.start, out.buffer);
        }
    }

    @FunctionTemplate(names = {"trunc", "truncate"}, scope = FunctionTemplate.FunctionScope.DECIMAL_SET_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseTruncateScaleFunction implements DrillSimpleFunc {

        @Param  Decimal38SparseHolder left;
        @Param  IntHolder right;
        @Output Decimal38SparseHolder result;
        public void setup() {

        }

        public void eval() {
          result.scale = right.value;
          result.precision = left.precision;
          result.buffer = left.buffer;
          result.start = left.start;
          boolean sign = left.getSign(left.start, left.buffer);

          int newScaleRoundedUp  = org.apache.drill.exec.util.DecimalUtility.roundUp(right.value);
          int origScaleRoundedUp = org.apache.drill.exec.util.DecimalUtility.roundUp(left.scale);

          if (right.value < left.scale) {
            // Get the source index beyond which we will truncate
            int srcIntIndex = 6 - origScaleRoundedUp - 1;
            int srcIndex = srcIntIndex + newScaleRoundedUp;

            // Truncate the remaining fractional part, move the integer part
            int destIndex = 6 - 1;
            if (srcIndex != destIndex) {
              while (srcIndex >= 0) {
                result.setInteger(destIndex--, result.getInteger(srcIndex--, result.start, result.buffer), result.start, result.buffer);
              }

              // Set the remaining portion of the decimal to be zeroes
              while (destIndex >= 0) {
                result.setInteger(destIndex--, 0, result.start, result.buffer);
              }
            }

            // We truncated the decimal digit. Now we need to truncate within the base 1 billion fractional digit
            int truncateFactor = org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS - (right.value % org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS);
            if (truncateFactor != org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS) {
              truncateFactor = (int) org.apache.drill.exec.util.DecimalUtility.getPowerOfTen(truncateFactor);
              int fractionalDigits = result.getInteger(6 - 1, result.start, result.buffer);
              fractionalDigits /= truncateFactor;
              result.setInteger(6 - 1, fractionalDigits * truncateFactor, result.start, result.buffer);
            }
          } else if (right.value > left.scale) {
            // Add fractional digits to the decimal

            // Check if we need to shift the decimal digits to the left
            if (newScaleRoundedUp > origScaleRoundedUp) {
              int srcIndex  = 0;
              int destIndex = newScaleRoundedUp - origScaleRoundedUp;

              // Check while extending scale, we are not overwriting integer part
              while (srcIndex < destIndex) {
                if (result.getInteger(srcIndex++, result.start, result.buffer) != 0) {
                  throw new org.apache.drill.common.exceptions.DrillRuntimeException("Truncate resulting in loss of integer part, reduce scale specified");
                }
              }

              srcIndex = 0;
              while (destIndex < 6) {
                result.setInteger(srcIndex++, result.getInteger(destIndex++, result.start, result.buffer), result.start, result.buffer);
              }

              // Clear the remaining part
              while (srcIndex < 6) {
                result.setInteger(srcIndex++, 0, result.start, result.buffer);
              }
            }
          }
          // Set the sign
          result.setSign(sign, result.start, result.buffer);
        }
    }

    @FunctionTemplate(name = "round", scope = FunctionTemplate.FunctionScope.DECIMAL_ZERO_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseRoundFunction implements DrillSimpleFunc {

        @Param  Decimal38SparseHolder in;
        @Output Decimal38SparseHolder out;
        public void setup() {

        }

        public void eval() {
          out.scale = 0;
          out.precision = in.precision;
          out.buffer = in.buffer;
          out.start = in.start;
          boolean sign = in.getSign(in.start, in.buffer);

          boolean roundUp = false;

          // Get the first fractional digit to see if want to round up or not
          int scaleIndex = 6 - org.apache.drill.exec.util.DecimalUtility.roundUp(in.scale);
          if (scaleIndex < 6) {
            int fractionalPart = out.getInteger(scaleIndex, out.start, out.buffer);
            int digit = fractionalPart / (org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE / 10);

            if (digit > 4) {
              roundUp = true;
            }
          }

          // Integer part's src index
          int srcIntIndex = scaleIndex - 1;

          // Truncate the fractional part, move the integer part
          int destIndex = 6 - 1;
          while (srcIntIndex >= 0) {
            out.setInteger(destIndex--, out.getInteger(srcIntIndex--, out.start, out.buffer), out.start, out.buffer);
          }

          // Set the remaining portion of the decimal to be zeroes
          while (destIndex >= 0) {
            out.setInteger(destIndex--, 0, out.start, out.buffer);
          }

          // Perform the roundup
          srcIntIndex = 6 - 1;
          if (roundUp == true) {
            while (srcIntIndex >= 0) {
              int value = out.getInteger(srcIntIndex, out.start, out.buffer) + 1;
              if (value >= org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE) {
                out.setInteger(srcIntIndex--, value % org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE, out.start, out.buffer);
                value = value / org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE;
              } else {
                out.setInteger(srcIntIndex--, value, out.start, out.buffer);
                break;
              }
            }
          }
          // Set the sign
          out.setSign(sign, out.start, out.buffer);
        }
    }

    @FunctionTemplate(name = "round", scope = FunctionTemplate.FunctionScope.DECIMAL_SET_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseRoundScaleFunction implements DrillSimpleFunc {

        @Param  Decimal38SparseHolder left;
        @Param  IntHolder right;
        @Output Decimal38SparseHolder result;
        public void setup() {

        }

        public void eval() {
          result.scale = right.value;
          result.precision = left.precision;
          result.buffer = left.buffer;
          result.start = left.start;
          boolean sign = left.getSign(left.start, left.buffer);

          org.apache.drill.exec.util.DecimalUtility.roundDecimal(result.buffer, result.start, result.nDecimalDigits, result.scale, left.scale);
          // Set the sign
          result.setSign(sign, result.start, result.buffer);
        }
    }


  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_HIGH,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareNullableDecimal38SparseVsNullableDecimal38SparseNullHigh implements DrillSimpleFunc {

    @Param NullableDecimal38SparseHolder left;
    @Param NullableDecimal38SparseHolder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
         outside:
          {

        if ( left.isSet == 0 ) {
          if ( right.isSet == 0 ) {
            out.value = 0;
            break outside;
          } else {
            out.value = 1;
            break outside;
          }
        } else if ( right.isSet == 0 ) {
          out.value = -1;
          break outside;
        }


            out.value = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
    }
  }



  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_LOW,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareNullableDecimal38SparseVsNullableDecimal38SparseNullLow implements DrillSimpleFunc {

    @Param NullableDecimal38SparseHolder left;
    @Param NullableDecimal38SparseHolder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
         outside:
          {

        if ( left.isSet == 0 ) {
          if ( right.isSet == 0 ) {
            out.value = 0;
            break outside;
          } else {
            out.value = -1;
            break outside;
          }
        } else if ( right.isSet == 0 ) {
          out.value = 1;
          break outside;
        }


            out.value = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
    }
  }


  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_HIGH,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareNullableDecimal38SparseVsDecimal38SparseNullHigh implements DrillSimpleFunc {

    @Param NullableDecimal38SparseHolder left;
    @Param Decimal38SparseHolder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
         outside:
          {

        if ( left.isSet == 0 ) {
          out.value = 1;
          break outside;
        }


            out.value = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
    }
  }



  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_LOW,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareNullableDecimal38SparseVsDecimal38SparseNullLow implements DrillSimpleFunc {

    @Param NullableDecimal38SparseHolder left;
    @Param Decimal38SparseHolder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
         outside:
          {

        if ( left.isSet == 0 ) {
          out.value = -1;
          break outside;
        }


            out.value = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
    }
  }


  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_HIGH,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareDecimal38SparseVsNullableDecimal38SparseNullHigh implements DrillSimpleFunc {

    @Param Decimal38SparseHolder left;
    @Param NullableDecimal38SparseHolder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
         outside:
          {

      if ( right.isSet == 0 ) {
        out.value = -1;
        break outside;
      }


            out.value = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
    }
  }



  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_LOW,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareDecimal38SparseVsNullableDecimal38SparseNullLow implements DrillSimpleFunc {

    @Param Decimal38SparseHolder left;
    @Param NullableDecimal38SparseHolder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
         outside:
          {

      if ( right.isSet == 0 ) {
        out.value = 1;
        break outside;
      }


            out.value = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
    }
  }


  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_HIGH,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareDecimal38SparseVsDecimal38SparseNullHigh implements DrillSimpleFunc {

    @Param Decimal38SparseHolder left;
    @Param Decimal38SparseHolder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
         outside:
          {



            out.value = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
    }
  }



  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_LOW,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareDecimal38SparseVsDecimal38SparseNullLow implements DrillSimpleFunc {

    @Param Decimal38SparseHolder left;
    @Param Decimal38SparseHolder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
         outside:
          {



            out.value = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
    }
  }



    @FunctionTemplate(name = "less than",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseLessThan implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {
            int cmp;
         outside:
          {


            cmp = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
            out.value = cmp == -1 ? 1 : 0;
        }
    }


    // TODO:  RESOLVE:  Here there are spaces in function template names, but
    // elsewhere there are underlines.  Are things being looked up correctly?
    @FunctionTemplate(name = "less than or equal to",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseLessThanEq implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {
            int cmp;
         outside:
          {


            cmp = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
            out.value = cmp < 1 ? 1 : 0;
        }
    }


    @FunctionTemplate(name = "greater than",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseGreaterThan implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {
            int cmp;
         outside:
          {


            cmp = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
            out.value = cmp == 1 ? 1 : 0;
        }
    }


    @FunctionTemplate(name = "greater than or equal to",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseGreaterThanEq implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {
            int cmp;
         outside:
          {


            cmp = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
            out.value = cmp > -1 ? 1 : 0;
        }
    }


    @FunctionTemplate(name = "Equal",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseEqual implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {
            int cmp;
         outside:
          {


            cmp = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
            out.value = cmp == 0 ? 1 : 0;
        }
    }


    @FunctionTemplate(name = "not equal",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal38SparseNotEqual implements DrillSimpleFunc {

        @Param Decimal38SparseHolder left;
        @Param Decimal38SparseHolder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {
            int cmp;
         outside:
          {


            cmp = org.apache.drill.exec.util.DecimalUtility.compareSparseBytes(left.buffer, left.start, left.getSign(left.start, left.buffer),
                            left.scale, left.precision, right.buffer,
                            right.start, right.getSign(right.start, right.buffer), right.precision,
                            right.scale, left.WIDTH, left.nDecimalDigits, false);

          } // outside
            out.value = cmp != 0 ? 1 : 0;
        }
    }
}




