/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.common.buffers;

import io.helidon.common.buffers.BufferData;
import io.helidon.common.buffers.DataListener;
import io.helidon.common.buffers.LazyString;
import io.helidon.common.buffers.ReadOnlyArrayData;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.function.Supplier;

public class DataReader {
    private final Supplier<byte[]> bytesSupplier;
    private final boolean ignoreLoneEol;
    private Node head;
    private Node tail;
    private DataListener listener;
    private Object context;

    public DataReader(Supplier<byte[]> bytesSupplier) {
        this.ignoreLoneEol = false;
        this.bytesSupplier = bytesSupplier;
        this.tail = this.head = new Node(BufferData.EMPTY_BYTES);
    }

    public DataReader(Supplier<byte[]> bytesSupplier, boolean ignoreLoneEol) {
        this.ignoreLoneEol = ignoreLoneEol;
        this.bytesSupplier = bytesSupplier;
        this.tail = this.head = new Node(BufferData.EMPTY_BYTES);
    }

    public int available() {
        int a = 0;
        Node n = this.head;
        while (n != null) {
            a += n.available();
            n = n.next;
        }
        return a;
    }

    public void pullData() {
        Node n;
        byte[] bytes = this.bytesSupplier.get();
        if (bytes == null) {
            throw new InsufficientDataAvailableException();
        }
        this.tail.next = n = new Node(bytes);
        this.tail = n;
    }

    public void skip(int lenToSkip) {
        while (lenToSkip > 0) {
            this.ensureAvailable();
            lenToSkip = this.head.skip(lenToSkip);
        }
    }

    public void ensureAvailable() {
        while (!this.head.hasAvailable()) {
            if (this.head.next == null) {
                this.pullData();
            }
            this.head = this.head.next;
        }
    }

    public byte read() {
        this.ensureAvailable();
        return this.head.bytes[this.head.position++];
    }

    public byte lookup() {
        this.ensureAvailable();
        return this.head.bytes[this.head.position];
    }

    public boolean startsWithNewLine() {
        this.ensureAvailable();
        byte[] bytes = this.head.bytes;
        int pos = this.head.position;
        return bytes[pos] == 13 && (pos + 1 < bytes.length ? bytes[pos + 1] : this.head.next().peek()) == 10;
    }

    public boolean startsWith(byte[] prefix) {
        this.ensureAvailable();
        if (prefix.length <= this.head.available()) {
            return Arrays.equals(this.head.bytes, this.head.position, this.head.position + prefix.length, prefix, 0, prefix.length);
        }
        int offset = 0;
        int remaining = prefix.length;
        Node n = this.head;
        while (remaining > 0) {
            int toCmp = Math.min(remaining, n.available());
            if (!Arrays.equals(n.bytes, n.position, n.position + toCmp, prefix, offset, offset + toCmp)) {
                return false;
            }
            offset += toCmp;
            if ((remaining -= toCmp) > 0 && n.next == null) {
                this.pullData();
            }
            n = n.next;
        }
        return true;
    }

    public BufferData readBuffer() {
        this.ensureAvailable();
        int size = this.head.available();
        BufferData result = BufferData.create(this.head.bytes, this.head.position, size);
        this.skip(size);
        return result;
    }

    public BufferData readBuffer(int length) {
        BufferData data = this.getBuffer(length);
        this.skip(length);
        return data;
    }

    public BufferData getBuffer(int length) {
        this.ensureAvailable();
        if (length <= this.head.available()) {
            return new ReadOnlyArrayData(this.head.bytes, this.head.position, length);
        }
        ArrayList<BufferData> data = new ArrayList<BufferData>();
        int remaining = length;
        Node n = this.head;
        while (remaining > 0) {
            int toAdd = Math.min(remaining, n.available());
            data.add(new ReadOnlyArrayData(n.bytes, n.position, toAdd));
            if ((remaining -= toAdd) > 0 && n.next == null) {
                this.pullData();
            }
            n = n.next;
        }
        return BufferData.create(data);
    }

    public LazyString readLazyString(Charset charset, int len) {
        this.ensureAvailable();
        if (len <= this.head.available()) {
            LazyString s = new LazyString(this.head.bytes, this.head.position, len, charset);
            this.head.position += len;
            return s;
        }
        byte[] b = new byte[len];
        int remaining = len;
        Node n = this.head;
        while (remaining > 0) {
            this.ensureAvailable();
            int toAdd = Math.min(remaining, n.available());
            System.arraycopy(n.bytes, n.position, b, len - remaining, toAdd);
            n.position += toAdd;
            if ((remaining -= toAdd) > 0 && n.next == null) {
                this.pullData();
            }
            n = n.next;
        }
        return new LazyString(b, charset);
    }

    public String readAsciiString(int len) {
        this.ensureAvailable();
        if (len <= this.head.available()) {
            String s = new String(this.head.bytes, this.head.position, len, StandardCharsets.US_ASCII);
            this.head.position += len;
            return s;
        }
        byte[] b = new byte[len];
        int remaining = len;
        Node n = this.head;
        while (remaining > 0) {
            this.ensureAvailable();
            int toAdd = Math.min(remaining, n.available());
            System.arraycopy(n.bytes, n.position, b, len - remaining, toAdd);
            n.position += toAdd;
            if ((remaining -= toAdd) > 0 && n.next == null) {
                this.pullData();
            }
            n = n.next;
        }
        return new String(b, StandardCharsets.US_ASCII);
    }

    public String readLine() throws IncorrectNewLineException {
        int i = this.findNewLine(Integer.MAX_VALUE);
        String s = this.readAsciiString(i);
        this.skip(2);
        return s;
    }

    public int findOrNewLine(byte b, int max) throws IncorrectNewLineException {
        this.ensureAvailable();
        int idx = 0;
        Node n = this.head;
        while (true) {
            byte[] barr = n.bytes;
            for (int i = n.position; i < barr.length && idx < max; ++i, ++idx) {
                if (barr[i] == 10 && !this.ignoreLoneEol) {
                    throw new IncorrectNewLineException("Found LF (" + idx + ") without preceding CR. :\n" + this.debugDataHex());
                }
                if (barr[i] == 13) {
                    byte nextByte = i + 1 < barr.length ? barr[i + 1] : n.next().peek();
                    if (nextByte == 10) {
                        return -idx - 1;
                    }
                    if (this.ignoreLoneEol) continue;
                    throw new IncorrectNewLineException("Found CR (" + idx + ") without following LF. :\n" + this.debugDataHex());
                }
                if (barr[i] != b) continue;
                return idx;
            }
            if (idx == max) {
                return max;
            }
            n = n.next();
        }
    }

    public String debugDataHex() {
        return this.getBuffer(this.available()).debugDataHex(true);
    }

    public int findNewLine(int max) throws IncorrectNewLineException {
        this.ensureAvailable();
        int idx = 0;
        Node n = this.head;
        while (true) {
            byte[] barr = n.bytes;
            for (int i = n.position; i < barr.length && idx < max; ++i, ++idx) {
                if (barr[i] == 10 && !this.ignoreLoneEol) {
                    throw new IncorrectNewLineException("Found LF (" + idx + ") without preceding CR. :\n" + this.debugDataHex());
                }
                if (barr[i] != 13) continue;
                byte nextByte = i + 1 < barr.length ? barr[i + 1] : n.next().peek();
                if (nextByte == 10) {
                    return idx;
                }
                if (this.ignoreLoneEol) continue;
                throw new IncorrectNewLineException("Found CR (" + idx + ") without following LF. :\n" + this.debugDataHex());
            }
            if (idx == max) {
                return max;
            }
            n = n.next();
        }
    }

    public <T> void listener(DataListener<T> listener, T context) {
        this.listener = listener;
        this.context = context;
    }

    private class Node {
        private final byte[] bytes;
        private int position;
        private Node next;

        Node(byte[] bytes) {
            this.bytes = bytes;
        }

        public String toString() {
            return this.position + " of " + Arrays.toString(this.bytes);
        }

        int available() {
            return this.bytes.length - this.position;
        }

        boolean hasAvailable() {
            return this.position < this.bytes.length;
        }

        int skip(int lenToSkip) {
            int newPos = this.position + lenToSkip;
            if (newPos <= this.bytes.length) {
                this.position = newPos;
                return 0;
            }
            this.position = this.bytes.length;
            return lenToSkip -= this.bytes.length - this.position;
        }

        Node next() {
            if (this.next == null) {
                assert (this == DataReader.this.tail);
                DataReader.this.pullData();
                assert (this.next != null);
            }
            return this.next;
        }

        byte peek() {
            return this.bytes[this.position];
        }
    }

    public static class InsufficientDataAvailableException
    extends RuntimeException {
    }

    public static class IncorrectNewLineException
    extends RuntimeException {
        public IncorrectNewLineException(String message) {
            super(message);
        }
    }
}

