/*
 * Decompiled with CFR 0.152.
 */
package io.activej.dns;

import io.activej.common.Checks;
import io.activej.common.StringFormatUtils;
import io.activej.common.initializer.WithInitializer;
import io.activej.common.time.CurrentTimeProvider;
import io.activej.dns.protocol.DnsProtocol;
import io.activej.dns.protocol.DnsQuery;
import io.activej.dns.protocol.DnsQueryException;
import io.activej.dns.protocol.DnsResponse;
import io.activej.eventloop.Eventloop;
import io.activej.jmx.api.attribute.JmxAttribute;
import io.activej.jmx.api.attribute.JmxOperation;
import io.activej.jmx.api.attribute.JmxReducers;
import io.activej.promise.Promise;
import java.net.InetAddress;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DnsCache
implements WithInitializer<DnsCache> {
    private static final Logger logger = LoggerFactory.getLogger(DnsCache.class);
    private static final boolean CHECK = Checks.isEnabled(DnsCache.class);
    public static final Duration DEFAULT_ERROR_CACHE_EXPIRATION = Duration.ofMinutes(1L);
    public static final Duration DEFAULT_TIMED_OUT_EXPIRATION = Duration.ofSeconds(1L);
    public static final Duration DEFAULT_HARD_EXPIRATION_DELTA = Duration.ofMinutes(1L);
    public static final Duration DEFAULT_MAX_TTL = null;
    private final Map<DnsQuery, CachedDnsQueryResult> cache = new ConcurrentHashMap<DnsQuery, CachedDnsQueryResult>();
    private final Eventloop eventloop;
    private long errorCacheExpiration = DEFAULT_ERROR_CACHE_EXPIRATION.toMillis();
    private long timedOutExpiration = DEFAULT_TIMED_OUT_EXPIRATION.toMillis();
    private long hardExpirationDelta = DEFAULT_HARD_EXPIRATION_DELTA.toMillis();
    private long maxTtl = Long.MAX_VALUE;
    private final AtomicBoolean cleaningUpNow = new AtomicBoolean(false);
    private final PriorityQueue<CachedDnsQueryResult> expirations = new PriorityQueue();
    @NotNull
    CurrentTimeProvider now;

    private DnsCache(@NotNull Eventloop eventloop) {
        this.eventloop = eventloop;
        this.now = eventloop;
    }

    public static DnsCache create(Eventloop eventloop) {
        return new DnsCache(eventloop);
    }

    public DnsCache withErrorCacheExpiration(Duration errorCacheExpiration) {
        this.errorCacheExpiration = errorCacheExpiration.toMillis();
        return this;
    }

    public DnsCache withTimedOutExpiration(Duration timedOutExpiration) {
        this.timedOutExpiration = timedOutExpiration.toMillis();
        return this;
    }

    public DnsCache withHardExpirationDelta(Duration hardExpirationDelta) {
        this.hardExpirationDelta = hardExpirationDelta.toMillis();
        return this;
    }

    public DnsCache withMaxTtl(@Nullable Duration maxTtl) {
        this.maxTtl = maxTtl == null ? Long.MAX_VALUE : maxTtl.toMillis();
        return this;
    }

    @Nullable
    public DnsQueryCacheResult tryToResolve(DnsQuery query) {
        CachedDnsQueryResult cachedResult = this.cache.get(query);
        if (cachedResult == null) {
            logger.trace("{} cache miss", (Object)query);
            return null;
        }
        DnsResponse result = cachedResult.response;
        assert (result != null);
        if (result.isSuccessful()) {
            logger.trace("{} cache hit", (Object)query);
        } else {
            logger.trace("{} error cache hit", (Object)query);
        }
        if (this.isExpired(cachedResult)) {
            logger.trace("{} hard TTL expired", (Object)query);
            return null;
        }
        if (this.isSoftExpired(cachedResult)) {
            logger.trace("{} soft TTL expired", (Object)query);
            return new DnsQueryCacheResult(result, true);
        }
        return new DnsQueryCacheResult(result, false);
    }

    private boolean isExpired(CachedDnsQueryResult cachedResult) {
        return this.now.currentTimeMillis() >= cachedResult.expirationTime + this.hardExpirationDelta;
    }

    private boolean isSoftExpired(CachedDnsQueryResult cachedResult) {
        return this.now.currentTimeMillis() >= cachedResult.expirationTime;
    }

    public void add(DnsQuery query, DnsResponse response) {
        if (CHECK) {
            Checks.checkState((boolean)this.eventloop.inEventloopThread(), (Object)"Concurrent cache adds are not allowed");
        }
        long expirationTime = this.now.currentTimeMillis();
        if (response.isSuccessful()) {
            assert (response.getRecord() != null);
            long minTtl = (long)response.getRecord().getMinTtl() * 1000L;
            if (minTtl == 0L) {
                return;
            }
            expirationTime += Math.min(minTtl, this.maxTtl);
        } else {
            expirationTime += response.getErrorCode() == DnsProtocol.ResponseErrorCode.TIMED_OUT ? this.timedOutExpiration : this.errorCacheExpiration;
        }
        CachedDnsQueryResult cachedResult = new CachedDnsQueryResult(response, expirationTime);
        CachedDnsQueryResult old = this.cache.put(query, cachedResult);
        this.expirations.add(cachedResult);
        if (old != null) {
            old.response = null;
            logger.trace("Refreshed cache entry for {}", (Object)query);
        } else {
            logger.trace("Added cache entry for {}", (Object)query);
        }
    }

    public void performCleanup() {
        CachedDnsQueryResult peeked;
        if (!this.cleaningUpNow.compareAndSet(false, true)) {
            return;
        }
        long currentTime = this.now.currentTimeMillis();
        while ((peeked = this.expirations.peek()) != null && peeked.expirationTime <= currentTime) {
            DnsResponse response = peeked.response;
            if (response != null) {
                DnsQuery query = response.getTransaction().getQuery();
                this.cache.remove(query);
                logger.trace("Cache entry expired for {}", (Object)query);
            }
            this.expirations.poll();
        }
        this.cleaningUpNow.set(false);
    }

    @JmxAttribute
    public Duration getErrorCacheExpiration() {
        return Duration.ofMillis(this.errorCacheExpiration);
    }

    @JmxAttribute
    public void setErrorCacheExpiration(Duration errorCacheExpiration) {
        this.errorCacheExpiration = errorCacheExpiration.toMillis();
    }

    @JmxAttribute
    public Duration getTimedOutExpiration() {
        return Duration.ofMillis(this.timedOutExpiration);
    }

    @JmxAttribute
    public void setTimedOutExpiration(Duration timedOutExpiration) {
        this.timedOutExpiration = timedOutExpiration.toMillis();
    }

    @JmxAttribute
    public Duration getHardExpirationDelta() {
        return Duration.ofMillis(this.hardExpirationDelta);
    }

    @JmxAttribute
    public void setHardExpirationDelta(Duration hardExpirationDelta) {
        this.hardExpirationDelta = hardExpirationDelta.toMillis();
    }

    @JmxAttribute
    public String getMaxTtl() {
        return this.maxTtl == Long.MAX_VALUE ? "" : StringFormatUtils.formatDuration((Duration)Duration.ofMillis(this.maxTtl));
    }

    @JmxAttribute
    public void setMaxTtl(String s) {
        this.maxTtl = s == null || s.isEmpty() ? Long.MAX_VALUE : StringFormatUtils.parseDuration((String)s).toMillis();
    }

    @JmxAttribute(reducer=JmxReducers.JmxReducerSum.class)
    public int getDomainsCount() {
        return this.cache.size();
    }

    @JmxAttribute(reducer=JmxReducers.JmxReducerSum.class)
    public int getFailedDomainsCount() {
        return (int)this.cache.values().stream().filter(cachedResult -> {
            assert (cachedResult.response != null);
            return !cachedResult.response.isSuccessful();
        }).count();
    }

    @JmxOperation
    public List<String> getResolvedDomains() {
        return this.getDomainNames(DnsResponse::isSuccessful);
    }

    @JmxOperation
    public List<String> getFailedDomains() {
        return this.getDomainNames(response -> !response.isSuccessful());
    }

    private List<String> getDomainNames(Predicate<DnsResponse> predicate) {
        return this.cache.entrySet().stream().filter(entry -> predicate.test(((CachedDnsQueryResult)entry.getValue()).response)).map(Map.Entry::getKey).map(DnsQuery::getDomainName).collect(Collectors.toList());
    }

    @JmxOperation
    public List<String> getDomainRecord(String domain) {
        return this.cache.entrySet().stream().filter(e -> ((DnsQuery)e.getKey()).getDomainName().equalsIgnoreCase(domain)).findFirst().map(e -> new RecordFormatter(((DnsQuery)e.getKey()).getDomainName(), (CachedDnsQueryResult)e.getValue()).formatMultiline()).orElse(Collections.emptyList());
    }

    @JmxOperation
    public List<String> getDomainRecords() {
        if (this.cache.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<String> lines = new ArrayList<String>(this.cache.size());
        lines.add("DomainName;Status;IP;MinTtlSeconds;SecondsToSoftExpiration;SecondsToHardExpiration");
        StringBuilder sb = new StringBuilder();
        this.cache.forEach((domainName, cachedResult) -> {
            RecordFormatter formatter = new RecordFormatter(domainName.getDomainName(), (CachedDnsQueryResult)cachedResult);
            lines.add(sb.append(formatter.domain).append(";").append((Object)formatter.getStatus()).append(";").append(formatter.getIps()).append(";").append(formatter.getMinTtlSeconds()).append(";").append(formatter.getSecondsToSoftExpiration()).append(";").append(formatter.getSecondsToHardExpiration()).toString());
            sb.setLength(0);
        });
        return lines;
    }

    @JmxOperation
    public void clear() {
        this.cache.clear();
        this.eventloop.submit(this.expirations::clear);
    }

    static final class CachedDnsQueryResult
    implements Comparable<CachedDnsQueryResult> {
        @Nullable
        DnsResponse response;
        final long expirationTime;

        CachedDnsQueryResult(@Nullable DnsResponse response, long expirationTime) {
            this.response = response;
            this.expirationTime = expirationTime;
        }

        @Override
        public int compareTo(CachedDnsQueryResult o) {
            return Long.compare(this.expirationTime, o.expirationTime);
        }
    }

    public static final class DnsQueryCacheResult {
        private final DnsResponse response;
        private final boolean needsRefreshing;
        @Nullable
        private DnsQueryException exception;

        public DnsQueryCacheResult(DnsResponse response, boolean needsRefreshing) {
            this.response = response;
            this.needsRefreshing = needsRefreshing;
        }

        public Promise<DnsResponse> getResponseAsPromise() {
            if (this.response.getErrorCode() == DnsProtocol.ResponseErrorCode.NO_ERROR) {
                return Promise.of((Object)this.response);
            }
            if (this.exception == null) {
                this.exception = new DnsQueryException(this.response);
            }
            return Promise.ofException((Exception)this.exception);
        }

        public boolean doesNeedRefreshing() {
            return this.needsRefreshing;
        }
    }

    private class RecordFormatter {
        final String domain;
        final CachedDnsQueryResult result;

        RecordFormatter(String domain, CachedDnsQueryResult result) {
            this.domain = domain;
            this.result = result;
        }

        DnsProtocol.ResponseErrorCode getStatus() {
            if (this.result.response == null) {
                return DnsProtocol.ResponseErrorCode.UNKNOWN;
            }
            return this.result.response.getErrorCode();
        }

        Collection<InetAddress> getIps() {
            if (this.result.response == null || this.result.response.getRecord() == null) {
                return Collections.emptyList();
            }
            return Arrays.asList(this.result.response.getRecord().getIps());
        }

        int getMinTtlSeconds() {
            if (this.result.response == null || this.result.response.getRecord() == null) {
                return 0;
            }
            return this.result.response.getRecord().getMinTtl();
        }

        String getSecondsToSoftExpiration() {
            long secs = (this.result.expirationTime - DnsCache.this.now.currentTimeMillis()) / 1000L;
            return this.formatExpired(secs);
        }

        String getSecondsToHardExpiration() {
            long secs = (this.result.expirationTime + DnsCache.this.hardExpirationDelta - DnsCache.this.now.currentTimeMillis()) / 1000L;
            return this.formatExpired(secs);
        }

        private String formatExpired(long secs) {
            return secs < 0L ? "expired" : Long.toString(secs);
        }

        public List<String> formatMultiline() {
            ArrayList<String> lines = new ArrayList<String>();
            lines.add("DomainName:\t" + this.domain);
            lines.add("Status:\t" + (Object)((Object)this.getStatus()));
            lines.add("IP:\t" + this.getIps());
            lines.add("MinTtlSeconds:\t" + this.getMinTtlSeconds());
            lines.add("SecondsToSoftExpiration:\t" + this.getSecondsToSoftExpiration());
            lines.add("SecondsToHardExpiration:\t" + this.getSecondsToHardExpiration());
            return lines;
        }
    }
}

