/*
 * Copyright (C) 2009 by Eric Herman <eric@freesa.org>
 * Copyright (C) 2009 by Robert Stewart <robert@wombatnation.com>
 * Copyright (C) 2013 by Eric Lambert <eric.d.lambert@gmail.com>
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in the parent directory for full text.
 */
package org.gearman.worker;

import java.util.List;
import java.util.Set;
import org.gearman.common.GearmanJobServerConnection;

/**
 *
 * A Gearman Worker is responsible for executing jobs it receives from the Job
 * Server. A Worker registers with the Job Server the types of jobs that it may
 * execute, the server will use this information, along with other attributes,
 * to determine which Worker will execute a particular job request. As data is
 * generated or as a job's state changes, the worker passes this information
 * back to the Job Server.
 */
public interface GearmanWorker {

    /**
     * This method is executed once the worker is ready to accept jobs from
     * the Gearman Job Server. It should be called after all the
     * {@link GearmanJobServerConnection} have been added to the worker and all
     * the applicable {@link GearmanFunction} have been registered with the
     * worker. After this method is executed, the Worker should be capable of
     * receiving jobs from the Gearman Job Server and executing thus jobs.
     *
     */
    void work();

    /**
     * Register a new Gearman Job Server with the worker.
     * @param conn {@link GearmanJobServerConnection} connected to the
     *          Gearman Job Server
     *
     * @return returns true if a connection to the server was established and
     *         the server was added to the worker, else false.
     *
     * @throws IllegalArgumentException If an invalid connection has been
     *         specified.
     *
     * @throws IllegalStateException If the worker has already been stopped.
     */
    boolean addServer(GearmanJobServerConnection conn);

    /**
     * Has a connection to the specified Gearman Job Server been registered with
     * this worker.
     *
     * @param conn The connection to the specified Gearman Job Server.
     *
     * @return  True if the Gearman Job Server has been registered with the worker,
     * otherwise false.
     */
    boolean hasServer(GearmanJobServerConnection conn);

    /**
     * Send an echo request to the specified {@link GearmanJobServerConnection},
     * the Gearman Job Server should echo the contents of the echo request back
     * to the Worker. This is can be used to verify the connection between a
     * Worker and a Server.
     *
     * @param text the text to include in the echo request.
     *
     * @param conn A connection to Gearman Job Server to which the request
     *             should be sent.
     *
     * @return a string which should be equal to text sent to the server.
     */
    String echo(String text, GearmanJobServerConnection conn);

    /**
     * Registers a particular {@link GearmanFunction} with the worker. Once a
     * function has been registered with a worker, the worker is capable of
     * executing any {@link org.gearman.client.GearmanJob} that matches the
     * registered function. Upon registering a function, the Worker notifies all
     * Gearman Job Servers that it can accept any job that contains the
     * applicable function.
     *
     * @param function The function being registered with the Worker.
     *
     * @param timeout The amount of time the Worker will allow a job to execute
     *                the particular function. If the execution of the function
     *                exceeds the timeout value, the job will be marked as
     *                failed.
     */
    void registerFunction(Class<? extends GearmanFunction> function, long timeout);

    /**
     * Registers a particular {@link GearmanFunction} with the worker. Once a
     * function has been registered with a worker, the worker is capable of
     * executing any {@link org.gearman.client.GearmanJob} that matches the
     * registered function. Upon registering a function, the Worker notifies all
     * Gearman Job Servers that is can accept any job that contains the
     * applicable function.
     *
     * @param function The function being registered with the Worker.
     */
    void registerFunction(Class<? extends GearmanFunction> function);

    /**
     * Registers a {@link GearmanFunctionFactory} with the worker and specifies
     * the timeout associated with any functions generated by this factory.
     *
     *
     * @param factory A function factory is factory that the worker will use to
     *                create the instance of a function when the worker needs a
     *                function of the type specified by the factory.
     *
     * @param timeout The amount of time the Worker will allow a job to execute
     *                the particular function. If the execution of the function
     *                exceeds the timeout value, the job will be marked as
     *                failed.
     */
    void registerFunctionFactory(GearmanFunctionFactory factory, long timeout);

    /**
     * Registers a {@link GearmanFunctionFactory} with the worker.
     *
     * @param factory A function factory is factory that the worker will use to
     *                create the instance of a function when the worker needs a
     *                function of the type specified by the factory.
     */
    void registerFunctionFactory(GearmanFunctionFactory factory);

    /**
     * Retrieve the names of all functions that have been registered with this
     * worker. If no functions have been registered, any empty set should be
     * returned.
     *
     * @return The name of all registered functions.
     */
    Set<String> getRegisteredFunctions();

    /**
     * Set the ID for a particular worker instance.  This enables monitoring and
     * reporting commands to uniquely identify specific workers.
     *
     * @param id The ID.
     */
    void setWorkerID(String id);

    /**
     * Set the ID used by a particular Gearman Job Server to identify a
     * particular Worker. This enables monitoring and reporting commands to
     * uniquely identify specific workers.
     *
     * @param id The ID.
     *
     * @param conn The Gearman Job Server to which this worker ID applies.
     */
    void setWorkerID(String id, GearmanJobServerConnection conn);

    /**
     * Retrieves the ID used by this worker instance.
     *
     * @return worker ID
     */
    String getWorkerID();

    /**
     * Retrieves the ID this worker uses to identify itself with a specific
     * connection.
     *
     * @param conn
     *
     * @return worker ID
     */
    String getWorkerID(GearmanJobServerConnection conn);

    /**
     * Unregisters a particular {@link GearmanFunction} from the worker. Once
     * a function has been unregistered from the Worker, a Worker will no longer
     * accept jobs which require the execution of the unregistered function.
     *
     * @param functionName
     */
    void unregisterFunction(String functionName);

    /**
     * Unregisters all{@link GearmanFunction} from the worker. The effect of
     * which is that the worker will not execute any new jobs.
     *
     */
    void unregisterAll();

    /**
     * Stops the work loop and requests a shutdown.
     */
    void stop();

    /**
     * Stops the work loop and closes all open connections.
     *
     * @return a List of any Exceptions thrown when closing connections
     */
    List<Exception> shutdown();

    /**
     * Determine if the worker is currently executing the work loop.
     *
     * @return True if the work loop is executing, else false.
     */
    boolean isRunning();

	/**
	 * Should worker only ask for jobs which contain a Unique Id.
	 * 
	 * @param requiresJobUUID
	 *            True if should worker only ask for jobs which contain a Unique
	 *            Id.
	 */
	void setJobUniqueIdRequired(boolean requiresJobUUID);

	/**
	 * Does worker only ask for jobs which contain a Unique Id.
	 * 
	 * @return true if worker only asks for jobs which contain a Unique Id, else
	 *         false.
	 */
	boolean isJobUniqueIdRequired();
}
